-- UDFs

-- SELECT * FROM stores
-- SELECT * FROM titles

--------------------------------------------------------------------------------------------------------------

-- Create a simple function that multiplies qty (of sales) by the title's price

--Just check for function existence
IF EXISTS (SELECT ROUTINE_NAME FROM INFORMATION_SCHEMA.ROUTINES
         WHERE ROUTINE_NAME = 'fn_GetTitleSalesValue')
   DROP FUNCTION fn_GetTitleSalesValue
GO

CREATE FUNCTION dbo.fn_GetTitleSalesValue (@qty INT, @title_ID VARCHAR(20)) -- (note in this instance 2 parameters)
RETURNS MONEY -- specificies the return type (must be a simple scalar)
AS
BEGIN
	RETURN (SELECT @qty * price AS SalesValue FROM titles
	WHERE title_id = @title_id)
END
GO

-- Try this amazing function out!
SELECT title_id, dbo.fn_GetTitleSalesValue (qty, title_id) AS SalesValue FROM sales

--------------------------------------------------------------------------------------------------------------

-- What about a simple function to return an author's full name
--Just check for function existence
IF EXISTS (SELECT ROUTINE_NAME FROM INFORMATION_SCHEMA.ROUTINES
         WHERE ROUTINE_NAME = 'fn_GetAuthorFullName')
   DROP FUNCTION fn_GetAuthorFullName
GO

CREATE FUNCTION dbo.fn_GetAuthorFullName (@author_id VARCHAR(20)) 
RETURNS VARCHAR(50)
AS
BEGIN
	RETURN (SELECT au_fname + ' ' + au_lname FROM authors
	WHERE au_id = @author_id)
END
GO

-- Now try it out!
SELECT dbo.fn_GetAuthorFullName(au_id) AS Author_Name FROM authors

--------------------------------------------------------------------------------------------------------------

-- SELECT * FROM stores
-- SELECT * FROM sales
-- SELECT * FROM titles
-- SELECT * FROM titleauthor
-- SELECT * FROM authors

-- What about an Inline Table-Valued function - a paramterised view?!!
-- What about a 'View' that shows what authors a particluar store sells and the values of the sales of that author
-- The query below is quite complex for an end-user to 'define' in a report and therefore a table function may help

SELECT st.stor_name AS Store_Name, dbo.fn_GetAuthorFullName(au.au_id) AS Author_Name,
SUM(sa.qty * t.price) AS Sales_Value
FROM stores st
JOIN sales sa ON st.stor_id = sa.stor_id
JOIN titles t ON t.title_id = sa.title_id
JOIN titleauthor ta ON ta.title_id = t.title_id
JOIN authors au ON au.au_id = ta.au_id
GROUP BY st.stor_name, dbo.fn_GetAuthorFullName(au.au_id)
ORDER BY Store_Name, Sales_Value DESC

-- the same query now created as a Inline Table-valued function that allows the end user to specify which stor_id they
-- want to look at sales for

IF EXISTS (SELECT ROUTINE_NAME FROM INFORMATION_SCHEMA.ROUTINES
         WHERE ROUTINE_NAME = 'fn_GetStoreSalesByAuthor')
   DROP FUNCTION fn_GetStoreSalesByAuthor
GO

CREATE FUNCTION dbo.fn_GetStoreSalesByAuthor (@store_id INT)
RETURNS TABLE
AS
RETURN
(
	SELECT TOP 100 st.stor_name AS Store_Name, dbo.fn_GetAuthorFullName(au.au_id) AS Author_Name,
	SUM(sa.qty * t.price) AS Sales_Value
	FROM stores st
	JOIN sales sa ON st.stor_id = sa.stor_id
	JOIN titles t ON t.title_id = sa.title_id
	JOIN titleauthor ta ON ta.title_id = t.title_id
	JOIN authors au ON au.au_id = ta.au_id
	WHERE st.stor_id = @store_id
	GROUP BY st.stor_name, dbo.fn_GetAuthorFullName(au.au_id)
	ORDER BY Sales_Value DESC
)
GO

-- OK let's try it out
SELECT * FROM dbo.fn_GetStoreSalesByAuthor (6380)

-- Note the syntax is NOT
-- SELECT dbo.fn_GetStoreSalesByAuthor (6380)

-- Obviously this query can be further restricted by using WHERE clauses
SELECT * FROM dbo.fn_GetStoreSalesByAuthor (6380)
WHERE Sales_Value > 50 --etc

--------------------------------------------------------------------------------------------------------------

-- What about a multi-statement table-valued UDF
-- All this means is that the result of the UDF is still a table
-- but that the result cannot be obtained by ONE simple select statement
-- and needs to be done through a set of statements

IF EXISTS (SELECT ROUTINE_NAME FROM INFORMATION_SCHEMA.ROUTINES
         WHERE ROUTINE_NAME = 'fn_Better_Sales_Days')
   DROP FUNCTION fn_Better_Sales_Days
GO


CREATE FUNCTION dbo.fn_Better_Sales_Days ()
RETURNS @sometable TABLE(order_date DATETIME, order_value MONEY)
AS

BEGIN

-- Now we want to use a cursor to look row by row at the result set to
-- determine upon which dates did sales exceed the previous sales 

-- Declare some variables
DECLARE @ord_value MONEY, @somedate DATETIME
DECLARE @last_ord_value MONEY

-- Set an initial value for the last order value
SET @last_ord_value = 0

-- Declare a simple cursor with the relevant select statement
DECLARE sales_cursor CURSOR
FOR SELECT SUM(qty * price) AS ord_value, ord_date
FROM sales s JOIN titles t ON s.title_id = t.title_id
GROUP BY ord_date
ORDER BY ord_date

-- Open the cursor and fetch the next row's data into the local variables
OPEN sales_cursor 
FETCH NEXT FROM sales_cursor INTO @ord_value, @somedate

-- Do some really fancy operations!!
IF @ord_value > @last_ord_value INSERT @sometable VALUES (@somedate, @ord_value)
SET @last_ord_value = @ord_value

-- set up a simple loop using one of SQL's in-built functions
WHILE @@FETCH_STATUS = 0 
BEGIN
FETCH NEXT FROM sales_cursor INTO @ord_value, @somedate

IF @ord_value > @last_ord_value INSERT @sometable VALUES (@somedate, @ord_value)
SET @last_ord_value = @ord_value

END
DEALLOCATE sales_cursor
RETURN
END
GO

--Try it out and see which days had sales greater than the previous ones

SELECT * FROM fn_Better_Sales_Days()
--------------------------------------------------------------------------------------------------------------


-- Note the problem with using Non-Deterministic routines inside UDFs
-- However, it doesn't stop you passing ND functions as arguments to UDFs

IF EXISTS (SELECT ROUTINE_NAME FROM INFORMATION_SCHEMA.ROUTINES
         WHERE ROUTINE_NAME = 'fn_WillNotWork_GetTommorow')
   DROP FUNCTION fn_WillNotWork_GetTommorow
GO

CREATE FUNCTION dbo.fn_WillNotWork_GetTommorow ()
RETURNS DATETIME
AS
BEGIN
	RETURN getdate() + 1
END
GO



IF EXISTS (SELECT ROUTINE_NAME FROM INFORMATION_SCHEMA.ROUTINES
         WHERE ROUTINE_NAME = 'fn_GetTommorow')
   DROP FUNCTION fn_GetTommorow
GO

CREATE FUNCTION dbo.fn_GetTommorow (@sometime DATETIME)
RETURNS DATETIME
AS
BEGIN
	RETURN @sometime + 1
END

-- This example shows the getdate function being used as the argument for a UDF
SELECT dbo.fn_GetTommorow (getdate())
--------------------------------------------------------------------------------------------------------------



